<?php
/**
 * Admin Dashboard - Message Form
 * Requires authentication - redirects to login if not authenticated
 */

require_once __DIR__ . '/auth.php';
require_once __DIR__ . '/../config/config.php';

// Require authentication (redirects to login if not authenticated)
requireAuth();

// Initialize database
$db = Database::getInstance();
$db->initDatabase();

// Get group count
$groupCount = $db->getGroupCount();
$groups = $db->getAllGroups();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard - Telegram Broadcast Bot</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }
        
        .container {
            max-width: 800px;
            margin: 0 auto;
            background: white;
            border-radius: 15px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.2);
            overflow: hidden;
        }
        
        .header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 30px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .header h1 {
            font-size: 28px;
            margin-bottom: 10px;
        }
        
        .header p {
            opacity: 0.9;
            font-size: 14px;
        }
        
        .logout-btn {
            background: rgba(255,255,255,0.2);
            color: white;
            border: 1px solid rgba(255,255,255,0.3);
            padding: 10px 20px;
            border-radius: 8px;
            text-decoration: none;
            font-size: 14px;
            transition: background 0.3s;
        }
        
        .logout-btn:hover {
            background: rgba(255,255,255,0.3);
        }
        
        .content {
            padding: 30px;
        }
        
        .stats {
            display: grid;
            grid-template-columns: 1fr;
            gap: 15px;
            margin-bottom: 25px;
        }
        
        .stat-card {
            background: #f8f9fa;
            padding: 20px;
            border-radius: 8px;
            text-align: center;
        }
        
        .stat-card h3 {
            font-size: 32px;
            color: #667eea;
            margin-bottom: 5px;
        }
        
        .stat-card p {
            color: #666;
            font-size: 14px;
        }
        
        .form-group {
            margin-bottom: 25px;
        }
        
        label {
            display: block;
            margin-bottom: 8px;
            color: #333;
            font-weight: 600;
            font-size: 14px;
        }
        
        textarea {
            width: 100%;
            padding: 15px;
            border: 2px solid #e0e0e0;
            border-radius: 8px;
            font-size: 15px;
            font-family: inherit;
            line-height: 1.6;
            resize: vertical;
            min-height: 200px;
            transition: border-color 0.3s;
        }
        
        textarea:focus {
            outline: none;
            border-color: #667eea;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
        }
        
        textarea::placeholder {
            color: #999;
        }
        
        .btn {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border: none;
            padding: 16px 30px;
            border-radius: 8px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            width: 100%;
            transition: transform 0.2s, box-shadow 0.2s;
            box-shadow: 0 4px 12px rgba(102, 126, 234, 0.3);
        }
        
        .btn:hover:not(:disabled) {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(102, 126, 234, 0.4);
        }
        
        .btn:active:not(:disabled) {
            transform: translateY(0);
        }
        
        .btn:disabled {
            opacity: 0.6;
            cursor: not-allowed;
            transform: none;
            box-shadow: none;
        }
        
        .loading {
            display: inline-block;
            width: 16px;
            height: 16px;
            border: 2px solid rgba(255,255,255,0.3);
            border-top-color: white;
            border-radius: 50%;
            animation: spin 0.8s linear infinite;
            margin-right: 8px;
            vertical-align: middle;
        }
        
        @keyframes spin {
            to { transform: rotate(360deg); }
        }
        
        .message {
            padding: 15px 20px;
            border-radius: 8px;
            margin-bottom: 25px;
            display: none;
            font-size: 15px;
            font-weight: 500;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
            animation: slideDown 0.3s ease-out;
        }
        
        @keyframes slideDown {
            from {
                opacity: 0;
                transform: translateY(-10px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        .message.success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
            border-left: 4px solid #28a745;
        }
        
        .message.error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
            border-left: 4px solid #dc3545;
        }
        
        .message.info {
            background: #d1ecf1;
            color: #0c5460;
            border: 1px solid #bee5eb;
            border-left: 4px solid #17a2b8;
        }
        
        .groups-list {
            margin-top: 25px;
            padding-top: 25px;
            border-top: 2px solid #e0e0e0;
        }
        
        .groups-list h3 {
            margin-bottom: 15px;
            color: #333;
        }
        
        .group-item {
            background: #f8f9fa;
            padding: 12px;
            border-radius: 6px;
            margin-bottom: 8px;
            font-size: 14px;
        }
        
        .group-item strong {
            color: #667eea;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <div>
                <h1>📢 Telegram Broadcast Panel</h1>
                <p>Send messages to all registered groups</p>
            </div>
            <a href="logout.php" class="logout-btn">Logout</a>
        </div>
        
        <div class="content">
            <div id="message" class="message"></div>
            
            <div class="stats" style="display: grid; grid-template-columns: repeat(3, 1fr); gap: 15px; margin-bottom: 25px;">
                <div class="stat-card">
                    <h3><?php echo $groupCount; ?></h3>
                    <p>Total Groups</p>
                </div>
                <div class="stat-card" style="background: #e7f3ff;">
                    <h3 style="color: #0066cc;">@pradicton_winloteery_bot</h3>
                    <p>Bot Username</p>
                </div>
                <div class="stat-card" style="background: #fff3cd;">
                    <h3 style="color: #856404;"><?php echo date('H:i'); ?></h3>
                    <p>Current Time</p>
                </div>
            </div>
            
            <!-- Quick Actions -->
            <div style="display: flex; gap: 10px; margin-bottom: 25px; flex-wrap: wrap;">
                <a href="auto_messages.php" class="btn" style="background: #28a745; flex: 1; text-align: center; text-decoration: none; display: flex; align-items: center; justify-content: center;">
                    ⏰ Auto Messages
                </a>
                <button onclick="refreshGroups()" class="btn" style="background: #17a2b8; flex: 1;">
                    🔄 Refresh Groups
                </button>
                <button onclick="deleteAllConfirm()" class="btn" style="background: #dc3545; flex: 1;">
                    🗑️ Delete All Groups
                </button>
            </div>
            
            <!-- Send to Specific Group by Link -->
            <div style="background: #f8f9fa; padding: 20px; border-radius: 8px; margin-bottom: 25px; border-left: 4px solid #28a745;">
                <h3 style="margin-bottom: 15px; color: #333;">📨 Send to Specific Group</h3>
                <p style="color: #666; font-size: 13px; margin-bottom: 15px;">Send message to a specific group by providing group link or ID. Bot must be added to the group first.</p>
                <form id="singleGroupForm">
                    <div class="form-group" style="margin-bottom: 15px;">
                        <label for="groupLink">Group Link or Group ID:</label>
                        <input 
                            type="text" 
                            id="groupLink" 
                            name="group_link" 
                            placeholder="Example: -1001234567890 or @groupname or https://t.me/c/1234567890/123"
                            style="width: 100%; padding: 12px; border: 2px solid #e0e0e0; border-radius: 8px; font-size: 14px; transition: border-color 0.3s;"
                            onfocus="this.style.borderColor='#28a745'"
                            onblur="this.style.borderColor='#e0e0e0'"
                        >
                        <small style="color: #666; font-size: 12px; display: block; margin-top: 5px;">
                            💡 Supported formats: Group ID (e.g., -1001234567890), @username, or t.me/c/ link
                        </small>
                    </div>
                    <div class="form-group" style="margin-bottom: 15px;">
                        <label for="singleMessageText">Message:</label>
                        <textarea 
                            id="singleMessageText" 
                            name="message" 
                            placeholder="Type your message here..."
                            required
                            rows="4"
                            style="width: 100%; padding: 12px; border: 2px solid #e0e0e0; border-radius: 8px; font-size: 14px; min-height: 100px;"
                        ></textarea>
                    </div>
                    <button type="submit" class="btn" id="sendSingleBtn" style="background: #28a745;">
                        <span id="singleBtnText">📤 Send to This Group</span>
                        <span id="singleBtnLoader" class="loading" style="display: none;"></span>
                    </button>
                </form>
            </div>
            
            <!-- Broadcast to All Groups -->
            <div style="background: #fff; padding: 20px; border-radius: 8px; margin-bottom: 25px; border: 2px solid #667eea;">
                <h3 style="margin-bottom: 15px; color: #333;">📢 Broadcast to All Groups</h3>
                <form id="broadcastForm" action="send.php" method="POST">
                    <div class="form-group">
                        <label for="messageText">Broadcast Message:</label>
                        <textarea 
                            id="messageText" 
                            name="message" 
                            placeholder="Type your message here... It will be sent to all registered groups."
                            required
                            rows="8"
                        ></textarea>
                    </div>
                    
                    <button type="submit" class="btn" id="sendBtn">
                        <span id="btnText">📤 Send to All Groups</span>
                        <span id="btnLoader" class="loading" style="display: none;"></span>
                    </button>
                </form>
            </div>
            
            <!-- Quick Message Templates -->
            <div style="background: #f0f8ff; padding: 20px; border-radius: 8px; margin-bottom: 25px; border-left: 4px solid #0066cc;">
                <h3 style="margin-bottom: 15px; color: #333;">⚡ Quick Templates</h3>
                <div style="display: grid; grid-template-columns: repeat(2, 1fr); gap: 10px;">
                    <button onclick="useTemplate('Hello! This is a broadcast message.')" class="btn" style="background: #6c757d; padding: 10px; font-size: 14px;">
                        📝 Simple Greeting
                    </button>
                    <button onclick="useTemplate('🎉 Welcome to our group! Stay tuned for updates.')" class="btn" style="background: #6c757d; padding: 10px; font-size: 14px;">
                        🎉 Welcome Message
                    </button>
                    <button onclick="useTemplate('📢 Important Announcement:\n\nPlease read carefully.')" class="btn" style="background: #6c757d; padding: 10px; font-size: 14px;">
                        📢 Announcement
                    </button>
                    <button onclick="useTemplate('⏰ Reminder: Don\'t forget to check updates!')" class="btn" style="background: #6c757d; padding: 10px; font-size: 14px;">
                        ⏰ Reminder
                    </button>
                </div>
            </div>
            
            <div class="groups-list">
                <h3>📋 Registered Groups</h3>
                <div id="groupsList">
                    <?php if (empty($groups)): ?>
                        <p style="color: #666;">No groups registered yet. Add the bot to a group to get started.</p>
                    <?php else: ?>
                        <?php foreach ($groups as $group): ?>
                            <div class="group-item" style="display: flex; justify-content: space-between; align-items: center;">
                                <div>
                                    <strong>Group ID:</strong> <?php echo htmlspecialchars($group['group_id']); ?>
                                </div>
                                <button 
                                    onclick="deleteGroup('<?php echo htmlspecialchars($group['group_id']); ?>')" 
                                    style="background: #dc3545; color: white; border: none; padding: 6px 12px; border-radius: 4px; cursor: pointer; font-size: 12px;"
                                    title="Delete this group"
                                >
                                    🗑️ Delete
                                </button>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
    
    <script>
        // Handle form submission with AJAX
        document.getElementById('broadcastForm').addEventListener('submit', async function(e) {
            e.preventDefault();
            
            const messageText = document.getElementById('messageText').value.trim();
            if (!messageText) {
                showMessage('Please enter a message', 'error');
                return;
            }
            
            const sendBtn = document.getElementById('sendBtn');
            const btnText = document.getElementById('btnText');
            const btnLoader = document.getElementById('btnLoader');
            const formData = new FormData(this);
            
            // Disable button and show loading state
            sendBtn.disabled = true;
            btnText.textContent = 'Sending to all groups...';
            btnLoader.style.display = 'inline-block';
            hideMessage();
            
            try {
                // Use absolute path to ensure correct endpoint
                const sendUrl = window.location.pathname.replace('dashboard.php', 'send.php');
                const response = await fetch(sendUrl, {
                    method: 'POST',
                    body: formData
                });
                
                // Check if response is OK
                if (!response.ok) {
                    throw new Error(`HTTP error! status: ${response.status}`);
                }
                
                const data = await response.json();
                
                if (data.success) {
                    // Success message
                    let successMsg = `✅ ${data.message}`;
                    if (data.sent && data.total) {
                        successMsg += ` (${data.sent}/${data.total} groups)`;
                    }
                    showMessage(successMsg, 'success');
                    
                    // Clear textarea
                    document.getElementById('messageText').value = '';
                    
                    // Focus back to textarea for next message
                    setTimeout(() => {
                        document.getElementById('messageText').focus();
                    }, 100);
                } else {
                    // Error message
                    showMessage(`❌ Error: ${data.message}`, 'error');
                }
            } catch (error) {
                // Network error or endpoint not found
                console.error('Broadcast error:', error);
                if (error.message.includes('404') || error.message.includes('not found')) {
                    showMessage(`❌ Error: Endpoint not found. Please check if send.php exists in admin folder.`, 'error');
                } else {
                    showMessage(`❌ Network error: ${error.message}. Please check your connection and try again.`, 'error');
                }
            } finally {
                // Re-enable button
                sendBtn.disabled = false;
                btnText.textContent = '📤 Send to All Groups';
                btnLoader.style.display = 'none';
            }
        });
        
        function showMessage(text, type) {
            const messageDiv = document.getElementById('message');
            messageDiv.textContent = text;
            messageDiv.className = `message ${type}`;
            messageDiv.style.display = 'block';
            
            // Auto-hide after 7 seconds for success, 10 seconds for errors
            const hideDelay = type === 'success' ? 7000 : 10000;
            setTimeout(hideMessage, hideDelay);
        }
        
        function hideMessage() {
            const messageDiv = document.getElementById('message');
            messageDiv.style.display = 'none';
            messageDiv.textContent = '';
        }
        
        // Allow manual close by clicking on message
        document.getElementById('message').addEventListener('click', function() {
            hideMessage();
        });
        
        // Use message template
        function useTemplate(template) {
            document.getElementById('messageText').value = template;
            document.getElementById('messageText').focus();
            // Scroll to broadcast form
            document.getElementById('messageText').scrollIntoView({ behavior: 'smooth', block: 'center' });
        }
        
        // Refresh groups list
        function refreshGroups() {
            location.reload();
        }
        
        // Delete single group
        async function deleteGroup(groupId) {
            if (!confirm(`Are you sure you want to delete group ${groupId}?`)) {
                return;
            }
            
            try {
                const formData = new FormData();
                formData.append('group_id', groupId);
                
                const response = await fetch('./delete_group.php', {
                    method: 'POST',
                    body: formData
                });
                
                const data = await response.json();
                
                if (data.success) {
                    showMessage(`✅ ${data.message}`, 'success');
                    setTimeout(() => location.reload(), 1000);
                } else {
                    showMessage(`❌ Error: ${data.message}`, 'error');
                }
            } catch (error) {
                showMessage(`❌ Error: ${error.message}`, 'error');
            }
        }
        
        // Delete all groups with confirmation
        function deleteAllConfirm() {
            if (!confirm('⚠️ Are you sure you want to delete ALL groups? This action cannot be undone!')) {
                return;
            }
            
            if (!confirm('⚠️ Final confirmation: Delete ALL groups?')) {
                return;
            }
            
            deleteAllGroups();
        }
        
        async function deleteAllGroups() {
            try {
                const response = await fetch('./delete_all.php', {
                    method: 'POST'
                });
                
                const data = await response.json();
                
                if (data.success) {
                    showMessage(`✅ ${data.message}`, 'success');
                    setTimeout(() => location.reload(), 1000);
                } else {
                    showMessage(`❌ Error: ${data.message}`, 'error');
                }
            } catch (error) {
                showMessage(`❌ Error: ${error.message}`, 'error');
            }
        }
        
        // Handle single group form submission
        document.getElementById('singleGroupForm').addEventListener('submit', async function(e) {
            e.preventDefault();
            
            const groupLink = document.getElementById('groupLink').value.trim();
            const messageText = document.getElementById('singleMessageText').value.trim();
            
            if (!groupLink) {
                showMessage('Please enter group link or group ID', 'error');
                return;
            }
            
            if (!messageText) {
                showMessage('Please enter a message', 'error');
                return;
            }
            
            const sendSingleBtn = document.getElementById('sendSingleBtn');
            const singleBtnText = document.getElementById('singleBtnText');
            const singleBtnLoader = document.getElementById('singleBtnLoader');
            
            sendSingleBtn.disabled = true;
            singleBtnText.textContent = 'Sending...';
            singleBtnLoader.style.display = 'inline-block';
            hideMessage();
            
            try {
                const formData = new FormData();
                formData.append('group_link', groupLink);
                formData.append('message', messageText);
                
                // Use relative path for send_single.php
                const sendUrl = './send_single.php';
                const response = await fetch(sendUrl, {
                    method: 'POST',
                    body: formData
                });
                
                if (!response.ok) {
                    throw new Error(`HTTP error! status: ${response.status}`);
                }
                
                const data = await response.json();
                
                if (data.success) {
                    showMessage(`✅ ${data.message}`, 'success');
                    document.getElementById('groupLink').value = '';
                    document.getElementById('singleMessageText').value = '';
                } else {
                    showMessage(`❌ Error: ${data.message}`, 'error');
                }
            } catch (error) {
                console.error('Single group send error:', error);
                if (error.message.includes('404') || error.message.includes('not found')) {
                    showMessage(`❌ Error: Endpoint not found. Please check if send_single.php exists.`, 'error');
                } else {
                    showMessage(`❌ Network error: ${error.message}`, 'error');
                }
            } finally {
                sendSingleBtn.disabled = false;
                singleBtnText.textContent = '📤 Send to This Group';
                singleBtnLoader.style.display = 'none';
            }
        });
    </script>
</body>
</html>
