<?php
/**
 * Broadcast Message Handler
 * Accepts POST message, fetches all group_ids, sends via Telegram API
 * Handles failed responses silently and respects rate limits
 */

require_once __DIR__ . '/auth.php';
require_once __DIR__ . '/../config/config.php';

header('Content-Type: application/json');

// Check if logged in (for API endpoint, return JSON instead of redirect)
if (!isAuthenticated()) {
    http_response_code(401);
    echo json_encode([
        'success' => false,
        'message' => 'Unauthorized - Please login'
    ]);
    exit;
}

// Accept POST message only
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode([
        'success' => false,
        'message' => 'Method not allowed'
    ]);
    exit;
}

// Get and sanitize message input
$message = $_POST['message'] ?? '';

// Basic input sanitization - trim whitespace
$message = trim($message);

// Validate message is not empty
if (empty($message)) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => 'Message is required'
    ]);
    exit;
}

// Basic input validation - max length check (Telegram limit is 4096 characters)
if (strlen($message) > 4096) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => 'Message is too long (max 4096 characters)'
    ]);
    exit;
}

try {
    // Initialize database
    $db = Database::getInstance();
    $db->initDatabase();
    
    // Fetch all group_ids
    $groups = $db->getAllGroups();
    
    if (empty($groups)) {
        echo json_encode([
            'success' => false,
            'message' => 'No groups registered'
        ]);
        exit;
    }
    
    // Telegram API endpoint
    $apiUrl = "https://api.telegram.org/bot" . BOT_TOKEN . "/sendMessage";
    
    $successCount = 0;
    $failedCount = 0;
    
    // Send message to each group
    foreach ($groups as $group) {
        $groupId = $group['group_id'];
        
        // Prepare request data with sanitized message
        // Note: Telegram API handles HTML escaping, but we ensure valid JSON
        $data = [
            'chat_id' => $groupId, // Group ID from database (already validated)
            'text' => $message,     // Sanitized message text
            'parse_mode' => 'HTML' // HTML parse mode for formatting
        ];
        
        // Make API request using cURL
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $apiUrl,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => json_encode($data),
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json'
            ],
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_TIMEOUT => 10
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        // Handle response silently
        if ($httpCode === 200) {
            $result = json_decode($response, true);
            if (isset($result['ok']) && $result['ok'] === true) {
                $successCount++;
            } else {
                // Failed response - handle silently (don't log or show)
                $failedCount++;
            }
        } else {
            // HTTP error - handle silently
            $failedCount++;
        }
        
        // Respect Telegram rate limits - basic loop delay
        // Telegram allows ~30 messages per second, so ~33ms delay is safe
        // Using 50ms for safety margin
        usleep(50000); // 50ms delay between requests
    }
    
    // Return success response
    $responseMessage = "Sent to {$successCount} groups";
    if ($failedCount > 0) {
        $responseMessage .= ", {$failedCount} failed";
    }
    
    echo json_encode([
        'success' => true,
        'message' => $responseMessage,
        'sent' => $successCount,
        'failed' => $failedCount,
        'total' => count($groups)
    ]);
    
} catch (Exception $e) {
    // Log error but don't expose details to user
    error_log("Broadcast error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Internal server error'
    ]);
}
