<?php
/**
 * Send Message to Single Group by Link/ID
 * Accepts group link or group ID and sends message immediately
 */

require_once __DIR__ . '/auth.php';
require_once __DIR__ . '/../config/config.php';

header('Content-Type: application/json');

// Check if logged in
if (!isAuthenticated()) {
    http_response_code(401);
    echo json_encode([
        'success' => false,
        'message' => 'Unauthorized - Please login'
    ]);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode([
        'success' => false,
        'message' => 'Method not allowed'
    ]);
    exit;
}

$groupLink = trim($_POST['group_link'] ?? '');
$message = trim($_POST['message'] ?? '');

if (empty($groupLink)) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => 'Group link or group ID is required'
    ]);
    exit;
}

if (empty($message)) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => 'Message is required'
    ]);
    exit;
}

// Validate message length
if (strlen($message) > 4096) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => 'Message is too long (max 4096 characters)'
    ]);
    exit;
}

try {
    // Extract group ID from link or use directly
    $groupId = extractGroupId($groupLink);
    
    if (!$groupId) {
        echo json_encode([
            'success' => false,
            'message' => 'Invalid group link or ID format. Please provide: Group ID (e.g., -1001234567890), @username, or t.me/c/ link. Note: Invite links (t.me/joinchat/) require group ID.'
        ]);
        exit;
    }
    
    // Initialize database
    $db = Database::getInstance();
    $db->initDatabase();
    
    // Add group to database if not exists (for future reference)
    $db->saveGroup($groupId);
    
    // Send message to the group
    $apiUrl = "https://api.telegram.org/bot" . BOT_TOKEN . "/sendMessage";
    
    $data = [
        'chat_id' => $groupId,
        'text' => $message,
        'parse_mode' => 'HTML'
    ];
    
    // Make API request
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $apiUrl,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => json_encode($data),
        CURLOPT_HTTPHEADER => [
            'Content-Type: application/json'
        ],
        CURLOPT_SSL_VERIFYPEER => true,
        CURLOPT_TIMEOUT => 10
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode === 200) {
        $result = json_decode($response, true);
        if (isset($result['ok']) && $result['ok'] === true) {
            echo json_encode([
                'success' => true,
                'message' => 'Message sent successfully to group!'
            ]);
        } else {
            $errorMsg = isset($result['description']) ? $result['description'] : 'Failed to send message';
            echo json_encode([
                'success' => false,
                'message' => 'Error: ' . $errorMsg
            ]);
        }
    } else {
        echo json_encode([
            'success' => false,
            'message' => 'HTTP error: ' . $httpCode
        ]);
    }
    
} catch (Exception $e) {
    error_log("Single group send error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Internal server error'
    ]);
}

/**
 * Extract group ID from various link formats
 * Supports: group ID, invite links, @username, t.me links
 * 
 * @param string $input Group link, ID, or username
 * @return string|null Group ID or username, or null if invalid
 */
function extractGroupId($input) {
    $input = trim($input);
    
    // If it's already a group ID (negative number for groups)
    if (preg_match('/^-?\d+$/', $input)) {
        return $input;
    }
    
    // If it's @username format (direct)
    if (preg_match('/^@(\w+)$/', $input, $matches)) {
        return $input; // Return @username as is
    }
    
    // Extract from t.me/c/chat_id/message_id format (channel/group with ID)
    // Format: https://t.me/c/1234567890/123
    if (preg_match('/t\.me\/c\/(\d+)/', $input, $matches)) {
        // Convert to negative group ID format (Telegram groups have negative IDs)
        $chatId = $matches[1];
        return '-' . $chatId;
    }
    
    // Extract from t.me/@username format
    if (preg_match('/t\.me\/(@?\w+)/', $input, $matches)) {
        $username = $matches[1];
        if (strpos($username, '@') !== 0) {
            $username = '@' . $username;
        }
        return $username;
    }
    
    // Extract from t.me/joinchat/ links (invite links)
    // Note: Invite links can't be directly converted to chat_id
    // But if bot is in group, we can try to get chat info
    if (preg_match('/t\.me\/joinchat\/([a-zA-Z0-9_-]+)/', $input, $matches)) {
        // For invite links, user needs to provide group ID manually
        // Or we can try to use getChat API if bot is member
        // For now, return null and show helpful error
        return null;
    }
    
    // If it contains @, treat as username
    if (strpos($input, '@') !== false && strpos($input, 't.me') === false) {
        if (strpos($input, '@') !== 0) {
            $input = '@' . ltrim($input, '@');
        }
        return $input;
    }
    
    // Try to extract any number that might be group ID
    if (preg_match('/(-\d+)/', $input, $matches)) {
        return $matches[1];
    }
    
    return null;
}
