<?php
/**
 * Database connection and helper functions
 * Reusable singleton database connection using PDO
 * All queries use prepared statements for security
 */

class Database {
    // Singleton pattern - single database connection instance
    private static $instance = null;
    private $connection;
    
    /**
     * Private constructor - prevents direct instantiation
     * Creates PDO connection with error handling
     */
    private function __construct() {
        try {
            // Build DSN (Data Source Name) for MySQL connection
            $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET;
            
            // PDO options for security and error handling
            $options = [
                PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,  // Throw exceptions on errors
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,        // Return associative arrays
                PDO::ATTR_EMULATE_PREPARES   => false,                   // Use real prepared statements
            ];
            
            // Create PDO connection using credentials from config.php
            $this->connection = new PDO($dsn, DB_USER, DB_PASS, $options);
        } catch (PDOException $e) {
            // Log error but don't expose database details
            error_log("Database connection failed: " . $e->getMessage());
            throw new Exception("Database connection failed");
        }
    }
    
    /**
     * Get singleton instance of Database
     * Reuses existing connection if available
     */
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Get PDO connection object
     * For advanced queries if needed
     */
    public function getConnection() {
        return $this->connection;
    }
    
    /**
     * Initialize database tables
     * Creates table if it doesn't exist
     * Safe to call multiple times
     */
    public function initDatabase() {
        try {
            $sql = "CREATE TABLE IF NOT EXISTS telegram_groups (
                id INT AUTO_INCREMENT PRIMARY KEY,
                group_id VARCHAR(50) UNIQUE NOT NULL
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
            
            $this->connection->exec($sql);
            
            // Create auto_messages table
            $sql2 = "CREATE TABLE IF NOT EXISTS auto_messages (
                id INT AUTO_INCREMENT PRIMARY KEY,
                message TEXT NOT NULL,
                send_time TIME NULL,
                send_date DATE NULL,
                start_time TIME NULL,
                end_time TIME NULL,
                start_date DATE NULL,
                end_date DATE NULL,
                interval_minutes INT NULL DEFAULT 1,
                frequency ENUM('daily', 'weekly', 'interval', 'once') NOT NULL DEFAULT 'daily',
                group_ids TEXT NULL,
                is_active TINYINT(1) NOT NULL DEFAULT 1,
                last_sent DATETIME NULL,
                created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
                updated_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
            
            $this->connection->exec($sql2);
            
            // Add new columns if they don't exist (for existing installations)
            try {
                $this->connection->exec("ALTER TABLE auto_messages ADD COLUMN start_time TIME NULL AFTER send_time");
            } catch (PDOException $e) {
                // Column might already exist, ignore
            }
            
            try {
                $this->connection->exec("ALTER TABLE auto_messages ADD COLUMN end_time TIME NULL AFTER start_time");
            } catch (PDOException $e) {
                // Column might already exist, ignore
            }
            
            try {
                $this->connection->exec("ALTER TABLE auto_messages ADD COLUMN send_date DATE NULL AFTER send_time");
            } catch (PDOException $e) {
                // Column might already exist, ignore
            }
            
            try {
                $this->connection->exec("ALTER TABLE auto_messages ADD COLUMN start_date DATE NULL AFTER start_time");
            } catch (PDOException $e) {
                // Column might already exist, ignore
            }
            
            try {
                $this->connection->exec("ALTER TABLE auto_messages ADD COLUMN end_date DATE NULL AFTER end_time");
            } catch (PDOException $e) {
                // Column might already exist, ignore
            }
            
            try {
                $this->connection->exec("ALTER TABLE auto_messages ADD COLUMN interval_minutes INT NULL DEFAULT 1 AFTER end_date");
            } catch (PDOException $e) {
                // Column might already exist, ignore
            }
            
            try {
                $this->connection->exec("ALTER TABLE auto_messages MODIFY COLUMN frequency ENUM('daily', 'weekly', 'interval', 'once') NOT NULL DEFAULT 'daily'");
            } catch (PDOException $e) {
                // Might fail if enum already includes interval, ignore
            }
            
            try {
                $this->connection->exec("ALTER TABLE auto_messages MODIFY COLUMN send_time TIME NULL");
            } catch (PDOException $e) {
                // Might fail, ignore
            }
            
            try {
                $this->connection->exec("ALTER TABLE auto_messages ADD COLUMN group_ids TEXT NULL AFTER frequency");
            } catch (PDOException $e) {
                // Column might already exist, ignore
            }
            
            // Create color_predictions table
            $sql3 = "CREATE TABLE IF NOT EXISTS color_predictions (
                id INT AUTO_INCREMENT PRIMARY KEY,
                result_type ENUM('big', 'small', 'number') NOT NULL,
                result_value VARCHAR(10) NOT NULL,
                period_number INT NOT NULL,
                is_sent TINYINT(1) NOT NULL DEFAULT 0,
                created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
                sent_at DATETIME NULL,
                INDEX idx_period (period_number),
                INDEX idx_sent (is_sent),
                INDEX idx_created (created_at)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
            
            $this->connection->exec($sql3);
            
            return true;
        } catch (PDOException $e) {
            error_log("Table creation failed: " . $e->getMessage());
            throw new Exception("Failed to create database table");
        }
    }
    
    /**
     * Save group information using INSERT IGNORE
     * Uses prepared statement to prevent SQL injection
     * INSERT IGNORE prevents duplicate entries
     * 
     * @param string $groupId Telegram group ID (negative number for groups)
     * @return bool True on success
     */
    public function saveGroup($groupId) {
        // INSERT IGNORE prevents errors if group_id already exists
        $sql = "INSERT IGNORE INTO telegram_groups (group_id) 
                VALUES (:group_id)";
        
        // Prepare statement for security (prevents SQL injection)
        $stmt = $this->connection->prepare($sql);
        $stmt->execute([
            ':group_id' => $groupId  // Bound parameter - safe from injection
        ]);
        
        return true;
    }
    
    /**
     * Get all registered groups from database
     * Returns array of groups with id and group_id
     * 
     * @return array Array of group records
     */
    public function getAllGroups() {
        $sql = "SELECT id, group_id 
                FROM telegram_groups 
                ORDER BY id DESC";  // Most recent first
        
        // Prepared statement for security
        $stmt = $this->connection->prepare($sql);
        $stmt->execute();
        
        // Fetch all results as associative array
        return $stmt->fetchAll();
    }
    
    /**
     * Get total count of registered groups
     * 
     * @return int Number of registered groups
     */
    public function getGroupCount() {
        $sql = "SELECT COUNT(*) as count FROM telegram_groups";
        
        // Prepared statement
        $stmt = $this->connection->prepare($sql);
        $stmt->execute();
        
        // Fetch single result
        $result = $stmt->fetch();
        return $result['count'];
    }
    
    /**
     * Delete a group by group_id
     * 
     * @param string $groupId Group ID to delete
     * @return bool True on success
     */
    public function deleteGroup($groupId) {
        $sql = "DELETE FROM telegram_groups WHERE group_id = :group_id";
        
        $stmt = $this->connection->prepare($sql);
        $stmt->execute([
            ':group_id' => $groupId
        ]);
        
        return $stmt->rowCount() > 0;
    }
    
    /**
     * Delete all groups
     * 
     * @return bool True on success
     */
    public function deleteAllGroups() {
        $sql = "DELETE FROM telegram_groups";
        $stmt = $this->connection->prepare($sql);
        $stmt->execute();
        return true;
    }
    
    /**
     * Auto Messages Methods
     */
    
    /**
     * Add a new auto message
     * 
     * @param string $message Message text
     * @param string|null $sendTime Time in HH:MM format (for daily/weekly/once)
     * @param string $frequency 'daily', 'weekly', 'interval', or 'once'
     * @param string|null $startTime Start time for interval (HH:MM format)
     * @param string|null $endTime End time for interval (HH:MM format)
     * @param string|null $sendDate Date for once frequency (YYYY-MM-DD format)
     * @param string|null $startDate Start date for interval (YYYY-MM-DD format)
     * @param string|null $endDate End date for interval (YYYY-MM-DD format)
     * @param int|null $intervalMinutes Interval in minutes (for interval frequency)
     * @return int|false Inserted ID or false on failure
     */
    public function addAutoMessage($message, $sendTime = null, $frequency = 'daily', $startTime = null, $endTime = null, $intervalMinutes = 1, $sendDate = null, $startDate = null, $endDate = null, $groupIds = null) {
        // Store group_ids as JSON string
        $groupIdsJson = null;
        if (!empty($groupIds) && is_array($groupIds)) {
            $groupIdsJson = json_encode($groupIds);
        } elseif (!empty($groupIds)) {
            $groupIdsJson = $groupIds;
        }
        
        $sql = "INSERT INTO auto_messages (message, send_time, send_date, start_time, end_time, start_date, end_date, interval_minutes, frequency, group_ids) 
                VALUES (:message, :send_time, :send_date, :start_time, :end_time, :start_date, :end_date, :interval_minutes, :frequency, :group_ids)";
        
        $stmt = $this->connection->prepare($sql);
        $stmt->execute([
            ':message' => $message,
            ':send_time' => $sendTime,
            ':send_date' => $sendDate,
            ':start_time' => $startTime,
            ':end_time' => $endTime,
            ':start_date' => $startDate,
            ':end_date' => $endDate,
            ':interval_minutes' => $intervalMinutes ?: 1,
            ':frequency' => $frequency,
            ':group_ids' => $groupIdsJson
        ]);
        
        return $this->connection->lastInsertId();
    }
    
    /**
     * Get all auto messages
     * 
     * @return array Array of auto message records
     */
    public function getAllAutoMessages() {
        $sql = "SELECT * FROM auto_messages ORDER BY send_time ASC, created_at DESC";
        
        $stmt = $this->connection->prepare($sql);
        $stmt->execute();
        
        return $stmt->fetchAll();
    }
    
    /**
     * Get auto message by ID
     * 
     * @param int $id Auto message ID
     * @return array|false Auto message record or false
     */
    public function getAutoMessage($id) {
        $sql = "SELECT * FROM auto_messages WHERE id = :id";
        
        $stmt = $this->connection->prepare($sql);
        $stmt->execute([':id' => $id]);
        
        return $stmt->fetch();
    }
    
    /**
     * Update auto message
     * 
     * @param int $id Auto message ID
     * @param string $message Message text
     * @param string|null $sendTime Time in HH:MM format (for daily/weekly/once)
     * @param string $frequency 'daily', 'weekly', 'interval', or 'once'
     * @param string|null $startTime Start time for interval (HH:MM format)
     * @param string|null $endTime End time for interval (HH:MM format)
     * @param string|null $sendDate Date for once frequency (YYYY-MM-DD format)
     * @param string|null $startDate Start date for interval (YYYY-MM-DD format)
     * @param string|null $endDate End date for interval (YYYY-MM-DD format)
     * @param int|null $intervalMinutes Interval in minutes (for interval frequency)
     * @return bool True on success
     */
    public function updateAutoMessage($id, $message, $sendTime = null, $frequency = 'daily', $startTime = null, $endTime = null, $intervalMinutes = 1, $sendDate = null, $startDate = null, $endDate = null, $groupIds = null) {
        // Store group_ids as JSON string
        $groupIdsJson = null;
        if (!empty($groupIds) && is_array($groupIds)) {
            $groupIdsJson = json_encode($groupIds);
        } elseif (!empty($groupIds)) {
            $groupIdsJson = $groupIds;
        }
        
        $sql = "UPDATE auto_messages 
                SET message = :message, send_time = :send_time, send_date = :send_date, 
                    start_time = :start_time, end_time = :end_time, 
                    start_date = :start_date, end_date = :end_date, 
                    interval_minutes = :interval_minutes, frequency = :frequency, group_ids = :group_ids
                WHERE id = :id";
        
        $stmt = $this->connection->prepare($sql);
        $stmt->execute([
            ':id' => $id,
            ':message' => $message,
            ':send_time' => $sendTime,
            ':send_date' => $sendDate,
            ':start_time' => $startTime,
            ':end_time' => $endTime,
            ':start_date' => $startDate,
            ':end_date' => $endDate,
            ':interval_minutes' => $intervalMinutes ?: 1,
            ':frequency' => $frequency,
            ':group_ids' => $groupIdsJson
        ]);
        
        return $stmt->rowCount() > 0;
    }
    
    /**
     * Delete auto message
     * 
     * @param int $id Auto message ID
     * @return bool True on success
     */
    public function deleteAutoMessage($id) {
        $sql = "DELETE FROM auto_messages WHERE id = :id";
        
        $stmt = $this->connection->prepare($sql);
        $stmt->execute([':id' => $id]);
        
        return $stmt->rowCount() > 0;
    }
    
    /**
     * Toggle auto message active status
     * 
     * @param int $id Auto message ID
     * @param int $isActive 1 for active, 0 for inactive
     * @return bool True on success
     */
    public function toggleAutoMessage($id, $isActive) {
        $sql = "UPDATE auto_messages SET is_active = :is_active WHERE id = :id";
        
        $stmt = $this->connection->prepare($sql);
        $stmt->execute([
            ':id' => $id,
            ':is_active' => $isActive
        ]);
        
        return $stmt->rowCount() > 0;
    }
    
    /**
     * Get auto messages that need to be sent now
     * Checks for daily messages, weekly messages, interval messages, and once messages
     * 
     * @return array Array of auto messages to send
     */
    public function getAutoMessagesToSend() {
        $currentTime = date('H:i:s');
        $currentDate = date('Y-m-d');
        $currentDateTime = date('Y-m-d H:i:s');
        
        $messagesToSend = [];
        
        // Get daily messages that are active and match current time
        // Compare by hour:minute to be more forgiving with cron timing
        $sql = "SELECT * FROM auto_messages 
                WHERE is_active = 1 
                AND frequency = 'daily' 
                AND TIME_FORMAT(send_time, '%H:%i') = TIME_FORMAT(:current_time, '%H:%i')
                AND (send_date IS NULL OR send_date <= :current_date)
                AND (last_sent IS NULL OR DATE(last_sent) != CURDATE())";
        
        $stmt = $this->connection->prepare($sql);
        $stmt->execute([
            ':current_time' => $currentTime,
            ':current_date' => $currentDate
        ]);
        $dailyMessages = $stmt->fetchAll();
        $messagesToSend = array_merge($messagesToSend, $dailyMessages);
        
        // Get weekly messages that are active, match current time
        $sql2 = "SELECT * FROM auto_messages 
                 WHERE is_active = 1 
                 AND frequency = 'weekly' 
                 AND TIME_FORMAT(send_time, '%H:%i') = TIME_FORMAT(:current_time, '%H:%i')
                 AND (send_date IS NULL OR send_date <= :current_date)
                 AND (last_sent IS NULL OR DATEDIFF(NOW(), last_sent) >= 7)";
        
        $stmt2 = $this->connection->prepare($sql2);
        $stmt2->execute([
            ':current_time' => $currentTime,
            ':current_date' => $currentDate
        ]);
        $weeklyMessages = $stmt2->fetchAll();
        $messagesToSend = array_merge($messagesToSend, $weeklyMessages);
        
        // Get once messages that match current date and time, and haven't been sent
        // For "once" messages, we check if date matches and time is within the same minute
        // This allows for slight timing differences in cron execution
        $sql4 = "SELECT * FROM auto_messages 
                 WHERE is_active = 1 
                 AND frequency = 'once' 
                 AND send_date = :current_date
                 AND TIME_FORMAT(send_time, '%H:%i') = TIME_FORMAT(:current_time, '%H:%i')
                 AND (last_sent IS NULL OR DATE(last_sent) != :current_date)";
        
        $stmt4 = $this->connection->prepare($sql4);
        $stmt4->execute([
            ':current_date' => $currentDate,
            ':current_time' => $currentTime
        ]);
        $onceMessages = $stmt4->fetchAll();
        $messagesToSend = array_merge($messagesToSend, $onceMessages);
        
        // Get interval messages that are active and within time range
        // Compare by hour:minute for time range check
        $sql3 = "SELECT * FROM auto_messages 
                 WHERE is_active = 1 
                 AND frequency = 'interval' 
                 AND start_time IS NOT NULL 
                 AND end_time IS NOT NULL
                 AND TIME_FORMAT(start_time, '%H:%i') <= TIME_FORMAT(:current_time, '%H:%i')
                 AND TIME_FORMAT(end_time, '%H:%i') >= TIME_FORMAT(:current_time, '%H:%i')
                 AND (start_date IS NULL OR start_date <= :current_date)
                 AND (end_date IS NULL OR end_date >= :current_date)";
        
        $stmt3 = $this->connection->prepare($sql3);
        $stmt3->execute([
            ':current_time' => $currentTime,
            ':current_date' => $currentDate
        ]);
        $intervalMessages = $stmt3->fetchAll();
        
        // Filter interval messages based on interval_minutes and last_sent
        foreach ($intervalMessages as $msg) {
            $intervalMinutes = intval($msg['interval_minutes'] ?: 1);
            
            // Check if we're within the date range
            if (!empty($msg['start_date']) && $msg['start_date'] > $currentDate) {
                continue; // Not yet started
            }
            
            if (!empty($msg['end_date']) && $msg['end_date'] < $currentDate) {
                continue; // Already ended
            }
            
            // If never sent, send it immediately
            if (empty($msg['last_sent'])) {
                $messagesToSend[] = $msg;
                continue;
            }
            
            // Calculate time difference in minutes
            $lastSent = new DateTime($msg['last_sent']);
            $now = new DateTime($currentDateTime);
            $diffMinutes = ($now->getTimestamp() - $lastSent->getTimestamp()) / 60;
            
            // If interval has passed, send it
            if ($diffMinutes >= $intervalMinutes) {
                $messagesToSend[] = $msg;
            }
        }
        
        return $messagesToSend;
    }
    
    /**
     * Update last_sent timestamp for auto message
     * 
     * @param int $id Auto message ID
     * @return bool True on success
     */
    public function updateAutoMessageLastSent($id) {
        $sql = "UPDATE auto_messages SET last_sent = NOW() WHERE id = :id";
        
        $stmt = $this->connection->prepare($sql);
        $stmt->execute([':id' => $id]);
        
        return $stmt->rowCount() > 0;
    }
    
    /**
     * Color Prediction Game Methods
     */
    
    /**
     * Generate and save color prediction result
     * 
     * @param string $type 'big', 'small', or 'number'
     * @param string $value The result value
     * @param int $periodNumber Period number
     * @return int|false Inserted ID or false on failure
     */
    public function addColorPrediction($type, $value, $periodNumber) {
        $sql = "INSERT INTO color_predictions (result_type, result_value, period_number) 
                VALUES (:type, :value, :period)";
        
        $stmt = $this->connection->prepare($sql);
        $stmt->execute([
            ':type' => $type,
            ':value' => $value,
            ':period' => $periodNumber
        ]);
        
        return $this->connection->lastInsertId();
    }
    
    /**
     * Get latest color prediction that hasn't been sent
     * 
     * @return array|false Prediction record or false
     */
    public function getLatestUnsentPrediction() {
        $sql = "SELECT * FROM color_predictions 
                WHERE is_sent = 0 
                ORDER BY created_at DESC 
                LIMIT 1";
        
        $stmt = $this->connection->prepare($sql);
        $stmt->execute();
        
        return $stmt->fetch();
    }
    
    /**
     * Mark prediction as sent
     * 
     * @param int $id Prediction ID
     * @return bool True on success
     */
    public function markPredictionAsSent($id) {
        $sql = "UPDATE color_predictions SET is_sent = 1, sent_at = NOW() WHERE id = :id";
        
        $stmt = $this->connection->prepare($sql);
        $stmt->execute([':id' => $id]);
        
        return $stmt->rowCount() > 0;
    }
    
    /**
     * Get current period number (based on minutes since epoch)
     * 
     * @return int Period number
     */
    public function getCurrentPeriodNumber() {
        // Period number = minutes since epoch (Jan 1, 1970)
        return intval(time() / 60);
    }
    
    /**
     * Check if prediction already exists for current period
     * 
     * @param int $periodNumber Period number
     * @return bool True if exists
     */
    public function predictionExistsForPeriod($periodNumber) {
        $sql = "SELECT COUNT(*) as count FROM color_predictions WHERE period_number = :period";
        
        $stmt = $this->connection->prepare($sql);
        $stmt->execute([':period' => $periodNumber]);
        
        $result = $stmt->fetch();
        return $result['count'] > 0;
    }
}
