<?php
/**
 * Telegram Bot API helper functions
 */

class TelegramBot {
    private $token;
    private $apiUrl;
    
    public function __construct($token = null) {
        $this->token = $token ?: BOT_TOKEN;
        $this->apiUrl = "https://api.telegram.org/bot" . $this->token;
    }
    
    /**
     * Send message to a chat
     */
    public function sendMessage($chatId, $text, $parseMode = 'HTML', $replyMarkup = null) {
        $url = $this->apiUrl . "/sendMessage";
        
        $data = [
            'chat_id' => $chatId,
            'text' => $text,
            'parse_mode' => $parseMode
        ];
        
        if ($replyMarkup !== null) {
            $data['reply_markup'] = $replyMarkup;
        }
        
        return $this->makeRequest($url, $data);
    }
    
    /**
     * Answer callback query (for inline keyboard buttons)
     */
    public function answerCallbackQuery($callbackQueryId, $text = '', $showAlert = false) {
        $url = $this->apiUrl . "/answerCallbackQuery";
        
        $data = [
            'callback_query_id' => $callbackQueryId,
            'text' => $text,
            'show_alert' => $showAlert
        ];
        
        return $this->makeRequest($url, $data);
    }
    
    /**
     * Set webhook
     * Compatible with: https://api.telegram.org/bot<TOKEN>/setWebhook?url=https://domain.com/webhook/bot.php
     * 
     * @param string $url Webhook URL (must be HTTPS)
     * @param string|null $secretToken Optional secret token for webhook verification
     * @return array API response
     */
    public function setWebhook($url, $secretToken = null) {
        $url_api = $this->apiUrl . "/setWebhook";
        
        $data = [
            'url' => $url
        ];
        
        if ($secretToken) {
            $data['secret_token'] = $secretToken;
        }
        
        return $this->makeRequest($url_api, $data);
    }
    
    /**
     * Set webhook using GET method (alternative method)
     * Compatible with: https://api.telegram.org/bot<TOKEN>/setWebhook?url=https://domain.com/webhook/bot.php
     * 
     * @param string $url Webhook URL (must be HTTPS)
     * @param string|null $secretToken Optional secret token for webhook verification
     * @return array API response
     */
    public function setWebhookGet($url, $secretToken = null) {
        $url_api = $this->apiUrl . "/setWebhook?url=" . urlencode($url);
        
        if ($secretToken) {
            $url_api .= "&secret_token=" . urlencode($secretToken);
        }
        
        return $this->makeRequest($url_api, [], 'GET');
    }
    
    /**
     * Delete webhook
     */
    public function deleteWebhook() {
        $url = $this->apiUrl . "/deleteWebhook";
        return $this->makeRequest($url, ['drop_pending_updates' => true]);
    }
    
    /**
     * Get webhook info
     */
    public function getWebhookInfo() {
        $url = $this->apiUrl . "/getWebhookInfo";
        return $this->makeRequest($url, [], 'GET');
    }
    
    /**
     * Get bot info (username, etc.)
     */
    public function getMe() {
        $url = $this->apiUrl . "/getMe";
        return $this->makeRequest($url, [], 'GET');
    }
    
    /**
     * Broadcast message to all groups
     */
    public function broadcast($message, $groups) {
        $results = [
            'success' => 0,
            'failed' => 0,
            'errors' => []
        ];
        
        foreach ($groups as $group) {
            $groupId = $group['group_id'];
            $result = $this->sendMessage($groupId, $message);
            
            if ($result && isset($result['ok']) && $result['ok']) {
                $results['success']++;
            } else {
                $results['failed']++;
                $error = isset($result['description']) ? $result['description'] : 'Unknown error';
                $results['errors'][] = [
                    'group_id' => $groupId,
                    'error' => $error
                ];
            }
            
            // Small delay to avoid rate limiting
            usleep(50000); // 50ms delay
        }
        
        return $results;
    }
    
    /**
     * Make HTTP request to Telegram API
     */
    private function makeRequest($url, $data = [], $method = 'POST') {
        $ch = curl_init();
        
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => $method === 'POST',
            CURLOPT_POSTFIELDS => $method === 'POST' ? json_encode($data) : null,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json'
            ],
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_TIMEOUT => 10
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        
        curl_close($ch);
        
        if ($error) {
            return ['ok' => false, 'error' => $error];
        }
        
        if ($httpCode !== 200) {
            return ['ok' => false, 'error' => "HTTP $httpCode"];
        }
        
        return json_decode($response, true);
    }
}
