<?php
/**
 * Cron Job Handler for Auto Messages
 * This script should be run every minute via cron job
 * 
 * Cron setup: * * * * * php /path/to/cron/send_auto_messages.php
 * Or: * * * * * curl -s https://yourdomain.com/cron/send_auto_messages.php > /dev/null
 */

// Set script execution time limit (5 minutes)
set_time_limit(300);

// Include configuration
require_once __DIR__ . '/../config/config.php';

// Log file for debugging (optional)
$logFile = __DIR__ . '/auto_messages.log';

function logMessage($message) {
    global $logFile;
    $timestamp = date('Y-m-d H:i:s');
    $logEntry = "[$timestamp] $message\n";
    file_put_contents($logFile, $logEntry, FILE_APPEND);
}

try {
    // Initialize database
    $db = Database::getInstance();
    $db->initDatabase();
    
    // Check for color prediction game messages
    $allAutoMessages = $db->getAllAutoMessages();
    $colorPredictionEnabled = false;
    $colorPredictionMsg = null;
    
    foreach ($allAutoMessages as $msg) {
        if (strpos($msg['message'], 'COLOR PREDICTION') !== false && $msg['is_active'] == 1) {
            $colorPredictionEnabled = true;
            $colorPredictionMsg = $msg;
            break;
        }
    }
    
    // Generate color prediction result if enabled
    if ($colorPredictionEnabled) {
        $currentPeriod = $db->getCurrentPeriodNumber();
        
        // Check if prediction already exists for this period
        if (!$db->predictionExistsForPeriod($currentPeriod)) {
            // Generate all three types
            $results = ColorPredictionGame::generateAllResults();
            
            // Rotate between big, small, number (based on period number)
            $types = ['big', 'small', 'number'];
            $selectedType = $types[$currentPeriod % 3];
            $selectedValue = $results[$selectedType];
            
            // Save prediction
            $predictionId = $db->addColorPrediction($selectedType, $selectedValue, $currentPeriod);
            
            if ($predictionId) {
                logMessage("Color prediction generated: Type=$selectedType, Value=$selectedValue, Period=$currentPeriod");
                
                // Format message
                $messageText = ColorPredictionGame::formatMessage($selectedType, $selectedValue, $currentPeriod);
                
                // Get selected groups
                $selectedGroupIds = [];
                if (!empty($colorPredictionMsg['group_ids'])) {
                    $selectedGroupIds = json_decode($colorPredictionMsg['group_ids'], true);
                    if (!is_array($selectedGroupIds)) {
                        $selectedGroupIds = [];
                    }
                }
                
                // Get all groups
                $allGroups = $db->getAllGroups();
                
                if (!empty($allGroups)) {
                    // Filter groups
                    $groupsToSend = [];
                    if (empty($selectedGroupIds)) {
                        $groupsToSend = $allGroups;
                    } else {
                        foreach ($allGroups as $group) {
                            if (in_array($group['group_id'], $selectedGroupIds)) {
                                $groupsToSend[] = $group;
                            }
                        }
                    }
                    
                    // Send to groups
                    $apiUrl = "https://api.telegram.org/bot" . BOT_TOKEN . "/sendMessage";
                    $successCount = 0;
                    $failedCount = 0;
                    
                    foreach ($groupsToSend as $group) {
                        $groupId = $group['group_id'];
                        
                        $data = [
                            'chat_id' => $groupId,
                            'text' => $messageText,
                            'parse_mode' => 'HTML'
                        ];
                        
                        $ch = curl_init();
                        curl_setopt_array($ch, [
                            CURLOPT_URL => $apiUrl,
                            CURLOPT_RETURNTRANSFER => true,
                            CURLOPT_POST => true,
                            CURLOPT_POSTFIELDS => json_encode($data),
                            CURLOPT_HTTPHEADER => [
                                'Content-Type: application/json'
                            ],
                            CURLOPT_SSL_VERIFYPEER => true,
                            CURLOPT_TIMEOUT => 10
                        ]);
                        
                        $response = curl_exec($ch);
                        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                        curl_close($ch);
                        
                        if ($httpCode === 200) {
                            $result = json_decode($response, true);
                            if (isset($result['ok']) && $result['ok'] === true) {
                                $successCount++;
                            } else {
                                $failedCount++;
                            }
                        } else {
                            $failedCount++;
                        }
                        
                        usleep(50000); // 50ms delay
                    }
                    
                    // Mark prediction as sent
                    $db->markPredictionAsSent($predictionId);
                    
                    logMessage("Color prediction sent: $successCount success, $failedCount failed");
                }
            }
        }
    }
    
    // Get regular messages that need to be sent
    $messagesToSend = $db->getAutoMessagesToSend();
    
    // Filter out color prediction messages (already handled above)
    $messagesToSend = array_filter($messagesToSend, function($msg) {
        return strpos($msg['message'], 'COLOR PREDICTION') === false;
    });
    
    if (empty($messagesToSend)) {
        // No messages to send at this time
        logMessage("No messages to send at " . date('H:i:s'));
        exit(0);
    }
    
    logMessage("Found " . count($messagesToSend) . " message(s) to send");
    
    // Get all groups
    $allGroups = $db->getAllGroups();
    
    if (empty($allGroups)) {
        logMessage("No groups registered - skipping message send");
        exit(0);
    }
    
    // Initialize Telegram bot
    $bot = new TelegramBot();
    
    // Telegram API endpoint
    $apiUrl = "https://api.telegram.org/bot" . BOT_TOKEN . "/sendMessage";
    
    // Process each message
    foreach ($messagesToSend as $autoMsg) {
        $messageText = $autoMsg['message'];
        $autoMsgId = $autoMsg['id'];
        
        // Get selected groups for this message
        $selectedGroupIds = [];
        if (!empty($autoMsg['group_ids'])) {
            $selectedGroupIds = json_decode($autoMsg['group_ids'], true);
            if (!is_array($selectedGroupIds)) {
                $selectedGroupIds = [];
            }
        }
        
        // If no groups selected, send to all groups
        $groupsToSend = [];
        if (empty($selectedGroupIds)) {
            $groupsToSend = $allGroups;
            logMessage("No specific groups selected for message ID $autoMsgId - sending to all groups");
        } else {
            // Filter groups to only selected ones
            foreach ($allGroups as $group) {
                if (in_array($group['group_id'], $selectedGroupIds)) {
                    $groupsToSend[] = $group;
                }
            }
            logMessage("Sending message ID $autoMsgId to " . count($groupsToSend) . " selected group(s)");
        }
        
        if (empty($groupsToSend)) {
            logMessage("No valid groups found for message ID $autoMsgId - skipping");
            continue;
        }
        
        if ($autoMsg['frequency'] === 'interval') {
            logMessage("Sending auto message ID: $autoMsgId (Frequency: interval, Range: {$autoMsg['start_time']} - {$autoMsg['end_time']}, Interval: {$autoMsg['interval_minutes']} min)");
        } else {
            logMessage("Sending auto message ID: $autoMsgId (Frequency: {$autoMsg['frequency']}, Time: {$autoMsg['send_time']})");
        }
        
        $successCount = 0;
        $failedCount = 0;
        
        // Send to each selected group
        foreach ($groupsToSend as $group) {
            $groupId = $group['group_id'];
            
            // Prepare request data
            $data = [
                'chat_id' => $groupId,
                'text' => $messageText,
                'parse_mode' => 'HTML'
            ];
            
            // Make API request using cURL
            $ch = curl_init();
            curl_setopt_array($ch, [
                CURLOPT_URL => $apiUrl,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_POST => true,
                CURLOPT_POSTFIELDS => json_encode($data),
                CURLOPT_HTTPHEADER => [
                    'Content-Type: application/json'
                ],
                CURLOPT_SSL_VERIFYPEER => true,
                CURLOPT_TIMEOUT => 10
            ]);
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);
            
            // Check response
            if ($httpCode === 200) {
                $result = json_decode($response, true);
                if (isset($result['ok']) && $result['ok'] === true) {
                    $successCount++;
                } else {
                    $failedCount++;
                    $error = isset($result['description']) ? $result['description'] : 'Unknown error';
                    logMessage("Failed to send to group $groupId: $error");
                }
            } else {
                $failedCount++;
                logMessage("HTTP error $httpCode for group $groupId");
            }
            
            // Respect Telegram rate limits
            usleep(50000); // 50ms delay between requests
        }
        
        // Update last_sent timestamp
        $db->updateAutoMessageLastSent($autoMsgId);
        
        logMessage("Auto message ID $autoMsgId sent: $successCount success, $failedCount failed");
    }
    
    logMessage("Cron job completed successfully");
    
} catch (Exception $e) {
    logMessage("ERROR: " . $e->getMessage());
    error_log("Auto messages cron error: " . $e->getMessage());
    exit(1);
}

exit(0);
