<?php
/**
 * Auto Messages System Test & Status
 * Ye script auto messages system ko test aur verify karta hai
 */

require_once __DIR__ . '/../config/config.php';

// Set execution time limit
set_time_limit(60);

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Auto Messages System Test</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }
        
        .container {
            max-width: 1000px;
            margin: 0 auto;
            background: white;
            border-radius: 15px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.2);
            overflow: hidden;
        }
        
        .header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 30px;
            text-align: center;
        }
        
        .header h1 {
            font-size: 32px;
            margin-bottom: 10px;
        }
        
        .content {
            padding: 30px;
        }
        
        .status-box {
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 20px;
            border-left: 5px solid;
        }
        
        .status-box.success {
            background: #d4edda;
            border-color: #28a745;
            color: #155724;
        }
        
        .status-box.error {
            background: #f8d7da;
            border-color: #dc3545;
            color: #721c24;
        }
        
        .status-box.warning {
            background: #fff3cd;
            border-color: #ffc107;
            color: #856404;
        }
        
        .status-box.info {
            background: #e7f3ff;
            border-color: #0066cc;
            color: #004085;
        }
        
        .test-section {
            background: #f8f9fa;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 20px;
        }
        
        .test-section h3 {
            color: #333;
            margin-bottom: 15px;
        }
        
        .btn {
            display: inline-block;
            padding: 12px 25px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            text-decoration: none;
            border-radius: 8px;
            font-weight: 600;
            margin: 10px 5px;
            border: none;
            cursor: pointer;
            font-size: 14px;
        }
        
        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(102, 126, 234, 0.3);
        }
        
        .btn-success {
            background: #28a745;
        }
        
        .btn-danger {
            background: #dc3545;
        }
        
        table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 15px;
            background: white;
        }
        
        th {
            background: #667eea;
            color: white;
            padding: 12px;
            text-align: left;
        }
        
        td {
            padding: 10px 12px;
            border-bottom: 1px solid #e0e0e0;
        }
        
        tr:hover {
            background: #f8f9fa;
        }
        
        .badge {
            display: inline-block;
            padding: 4px 10px;
            border-radius: 12px;
            font-size: 12px;
            font-weight: 600;
        }
        
        .badge-success {
            background: #28a745;
            color: white;
        }
        
        .badge-danger {
            background: #dc3545;
            color: white;
        }
        
        .badge-warning {
            background: #ffc107;
            color: #333;
        }
        
        code {
            background: #f8f9fa;
            padding: 2px 6px;
            border-radius: 4px;
            font-family: 'Courier New', monospace;
            font-size: 13px;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>🧪 Auto Messages System Test</h1>
            <p>System Status & Testing</p>
        </div>
        
        <div class="content">
            <?php
            try {
                // Initialize database
                $db = Database::getInstance();
                $db->initDatabase();
                
                echo '<div class="status-box success">';
                echo '✅ <strong>Database Connection:</strong> Successful!';
                echo '</div>';
                
                // Test 1: Database Tables
                echo '<div class="test-section">';
                echo '<h3>📊 Database Tables Check</h3>';
                
                $tables = ['telegram_groups', 'auto_messages', 'color_predictions'];
                $allTablesExist = true;
                
                foreach ($tables as $table) {
                    try {
                        $sql = "SHOW TABLES LIKE '$table'";
                        $stmt = $db->getConnection()->prepare($sql);
                        $stmt->execute();
                        $exists = $stmt->rowCount() > 0;
                        
                        if ($exists) {
                            echo '<div class="status-box success" style="margin-bottom: 10px;">';
                            echo "✅ Table <code>$table</code> exists";
                            echo '</div>';
                        } else {
                            echo '<div class="status-box error" style="margin-bottom: 10px;">';
                            echo "❌ Table <code>$table</code> missing!";
                            echo '</div>';
                            $allTablesExist = false;
                        }
                    } catch (Exception $e) {
                        echo '<div class="status-box error" style="margin-bottom: 10px;">';
                        echo "❌ Error checking table <code>$table</code>: " . htmlspecialchars($e->getMessage());
                        echo '</div>';
                        $allTablesExist = false;
                    }
                }
                echo '</div>';
                
                // Test 2: Groups
                echo '<div class="test-section">';
                echo '<h3>👥 Registered Groups</h3>';
                
                $allGroups = $db->getAllGroups();
                $groupCount = count($allGroups);
                
                if ($groupCount > 0) {
                    echo '<div class="status-box success">';
                    echo "✅ <strong>Total Groups:</strong> $groupCount";
                    echo '</div>';
                    
                    echo '<table>';
                    echo '<tr><th>ID</th><th>Group ID</th></tr>';
                    foreach ($allGroups as $group) {
                        echo '<tr>';
                        echo '<td>' . htmlspecialchars($group['id']) . '</td>';
                        echo '<td><code>' . htmlspecialchars($group['group_id']) . '</code></td>';
                        echo '</tr>';
                    }
                    echo '</table>';
                } else {
                    echo '<div class="status-box warning">';
                    echo "⚠️ <strong>No groups registered yet!</strong> Bot ko groups mein add karo.";
                    echo '</div>';
                }
                echo '</div>';
                
                // Test 3: Auto Messages
                echo '<div class="test-section">';
                echo '<h3>📨 Auto Messages Status</h3>';
                
                $autoMessages = $db->getAllAutoMessages();
                $activeMessages = array_filter($autoMessages, function($msg) {
                    return $msg['is_active'] == 1;
                });
                
                echo '<div class="status-box info">';
                echo "📊 <strong>Total Messages:</strong> " . count($autoMessages) . " | ";
                echo "<strong>Active:</strong> " . count($activeMessages);
                echo '</div>';
                
                if (!empty($autoMessages)) {
                    echo '<table>';
                    echo '<tr><th>ID</th><th>Message</th><th>Frequency</th><th>Status</th><th>Last Sent</th></tr>';
                    foreach ($autoMessages as $msg) {
                        $statusClass = $msg['is_active'] == 1 ? 'badge-success' : 'badge-danger';
                        $statusText = $msg['is_active'] == 1 ? 'Active' : 'Inactive';
                        
                        $messagePreview = substr($msg['message'], 0, 50);
                        if (strlen($msg['message']) > 50) {
                            $messagePreview .= '...';
                        }
                        
                        echo '<tr>';
                        echo '<td>' . htmlspecialchars($msg['id']) . '</td>';
                        echo '<td>' . htmlspecialchars($messagePreview) . '</td>';
                        echo '<td>' . htmlspecialchars(ucfirst($msg['frequency'])) . '</td>';
                        echo '<td><span class="badge ' . $statusClass . '">' . $statusText . '</span></td>';
                        echo '<td>' . ($msg['last_sent'] ? htmlspecialchars($msg['last_sent']) : 'Never') . '</td>';
                        echo '</tr>';
                    }
                    echo '</table>';
                } else {
                    echo '<div class="status-box warning">';
                    echo "⚠️ <strong>No auto messages configured!</strong>";
                    echo '</div>';
                }
                echo '</div>';
                
                // Test 4: Color Prediction Status
                echo '<div class="test-section">';
                echo '<h3>🎯 Color Prediction Game Status</h3>';
                
                $colorPredictionEnabled = false;
                $colorPredictionMsg = null;
                foreach ($autoMessages as $msg) {
                    if (strpos($msg['message'], 'COLOR PREDICTION') !== false) {
                        $colorPredictionEnabled = $msg['is_active'] == 1;
                        $colorPredictionMsg = $msg;
                        break;
                    }
                }
                
                if ($colorPredictionEnabled) {
                    echo '<div class="status-box success">';
                    echo "✅ <strong>Color Prediction Game:</strong> ENABLED";
                    echo '</div>';
                    
                    if ($colorPredictionMsg) {
                        echo '<div class="status-box info">';
                        echo "<strong>Start Time:</strong> " . ($colorPredictionMsg['start_time'] ?: 'N/A') . "<br>";
                        echo "<strong>End Time:</strong> " . ($colorPredictionMsg['end_time'] ?: 'N/A') . "<br>";
                        echo "<strong>Interval:</strong> " . ($colorPredictionMsg['interval_minutes'] ?: 1) . " minute(s)";
                        echo '</div>';
                    }
                } else {
                    echo '<div class="status-box warning">';
                    echo "⚠️ <strong>Color Prediction Game:</strong> DISABLED";
                    echo '<br><small>Admin panel se enable karo: <code>admin/auto_messages.php</code></small>';
                    echo '</div>';
                }
                
                // Check recent predictions
                try {
                    $sql = "SELECT * FROM color_predictions ORDER BY created_at DESC LIMIT 10";
                    $stmt = $db->getConnection()->prepare($sql);
                    $stmt->execute();
                    $recentPredictions = $stmt->fetchAll();
                    
                    if (!empty($recentPredictions)) {
                        echo '<h4 style="margin-top: 20px; margin-bottom: 10px;">Recent Predictions:</h4>';
                        echo '<table>';
                        echo '<tr><th>Period</th><th>Type</th><th>Value</th><th>Sent</th><th>Created</th></tr>';
                        foreach ($recentPredictions as $pred) {
                            $sentBadge = $pred['is_sent'] == 1 ? 'badge-success' : 'badge-warning';
                            $sentText = $pred['is_sent'] == 1 ? 'Yes' : 'No';
                            
                            echo '<tr>';
                            echo '<td>#' . htmlspecialchars($pred['period_number']) . '</td>';
                            echo '<td>' . htmlspecialchars(strtoupper($pred['result_type'])) . '</td>';
                            echo '<td><strong>' . htmlspecialchars($pred['result_value']) . '</strong></td>';
                            echo '<td><span class="badge ' . $sentBadge . '">' . $sentText . '</span></td>';
                            echo '<td>' . htmlspecialchars($pred['created_at']) . '</td>';
                            echo '</tr>';
                        }
                        echo '</table>';
                    }
                } catch (Exception $e) {
                    // Table might not exist yet
                }
                
                echo '</div>';
                
                // Test 5: Messages Ready to Send
                echo '<div class="test-section">';
                echo '<h3>📤 Messages Ready to Send</h3>';
                
                $messagesToSend = $db->getAutoMessagesToSend();
                
                // Filter out color prediction messages (they're handled separately)
                $regularMessages = array_filter($messagesToSend, function($msg) {
                    return strpos($msg['message'], 'COLOR PREDICTION') === false;
                });
                
                if (!empty($regularMessages)) {
                    echo '<div class="status-box success">';
                    echo "✅ <strong>" . count($regularMessages) . " message(s)</strong> ready to send right now!";
                    echo '</div>';
                    
                    echo '<table>';
                    echo '<tr><th>ID</th><th>Message</th><th>Frequency</th></tr>';
                    foreach ($regularMessages as $msg) {
                        $messagePreview = substr($msg['message'], 0, 60);
                        if (strlen($msg['message']) > 60) {
                            $messagePreview .= '...';
                        }
                        
                        echo '<tr>';
                        echo '<td>' . htmlspecialchars($msg['id']) . '</td>';
                        echo '<td>' . htmlspecialchars($messagePreview) . '</td>';
                        echo '<td>' . htmlspecialchars(ucfirst($msg['frequency'])) . '</td>';
                        echo '</tr>';
                    }
                    echo '</table>';
                } else {
                    echo '<div class="status-box info">';
                    echo "ℹ️ <strong>No regular messages</strong> ready to send at this time.";
                    if ($colorPredictionEnabled) {
                        echo '<br><small>Color prediction messages are handled separately by cron job.</small>';
                    }
                    echo '</div>';
                }
                echo '</div>';
                
                // Test 6: Cron Job Test
                echo '<div class="test-section">';
                echo '<h3>⏰ Cron Job Test</h3>';
                
                $cronFile = __DIR__ . '/send_auto_messages.php';
                $cronExists = file_exists($cronFile);
                
                if ($cronExists) {
                    echo '<div class="status-box success">';
                    echo "✅ <strong>Cron script exists:</strong> <code>cron/send_auto_messages.php</code>";
                    echo '</div>';
                    
                    // Check log file
                    $logFile = __DIR__ . '/auto_messages.log';
                    if (file_exists($logFile)) {
                        $logSize = filesize($logFile);
                        $logModified = date('Y-m-d H:i:s', filemtime($logFile));
                        
                        echo '<div class="status-box info">';
                        echo "📝 <strong>Log File:</strong> <code>cron/auto_messages.log</code><br>";
                        echo "<strong>Size:</strong> " . number_format($logSize / 1024, 2) . " KB<br>";
                        echo "<strong>Last Modified:</strong> $logModified";
                        echo '</div>';
                        
                        // Show last 5 log entries
                        if ($logSize > 0) {
                            $logContent = file_get_contents($logFile);
                            $logLines = explode("\n", $logContent);
                            $lastLines = array_slice(array_filter($logLines), -5);
                            
                            if (!empty($lastLines)) {
                                echo '<h4 style="margin-top: 15px; margin-bottom: 10px;">Last 5 Log Entries:</h4>';
                                echo '<div style="background: #2d2d2d; color: #f8f8f2; padding: 15px; border-radius: 8px; font-family: monospace; font-size: 12px; max-height: 200px; overflow-y: auto;">';
                                foreach ($lastLines as $line) {
                                    echo htmlspecialchars($line) . '<br>';
                                }
                                echo '</div>';
                            }
                        }
                    } else {
                        echo '<div class="status-box warning">';
                        echo "⚠️ <strong>Log file not created yet.</strong> Cron job run hone ke baad create hogi.";
                        echo '</div>';
                    }
                } else {
                    echo '<div class="status-box error">';
                    echo "❌ <strong>Cron script not found!</strong>";
                    echo '</div>';
                }
                
                // Test 7: Telegram API Connection
                echo '<div class="test-section">';
                echo '<h3>🤖 Telegram API Test</h3>';
                
                try {
                    $apiUrl = "https://api.telegram.org/bot" . BOT_TOKEN . "/getMe";
                    $ch = curl_init();
                    curl_setopt_array($ch, [
                        CURLOPT_URL => $apiUrl,
                        CURLOPT_RETURNTRANSFER => true,
                        CURLOPT_SSL_VERIFYPEER => true,
                        CURLOPT_TIMEOUT => 10
                    ]);
                    
                    $response = curl_exec($ch);
                    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                    curl_close($ch);
                    
                    if ($httpCode === 200) {
                        $result = json_decode($response, true);
                        if (isset($result['ok']) && $result['ok'] === true) {
                            $botInfo = $result['result'];
                            echo '<div class="status-box success">';
                            echo "✅ <strong>Telegram API:</strong> Connected!<br>";
                            echo "<strong>Bot Name:</strong> " . htmlspecialchars($botInfo['first_name']) . "<br>";
                            echo "<strong>Bot Username:</strong> @" . htmlspecialchars($botInfo['username'] ?? 'N/A') . "<br>";
                            echo "<strong>Bot ID:</strong> " . htmlspecialchars($botInfo['id']);
                            echo '</div>';
                        } else {
                            echo '<div class="status-box error">';
                            echo "❌ <strong>Telegram API Error:</strong> " . htmlspecialchars($result['description'] ?? 'Unknown error');
                            echo '</div>';
                        }
                    } else {
                        echo '<div class="status-box error">';
                        echo "❌ <strong>HTTP Error:</strong> $httpCode";
                        echo '</div>';
                    }
                } catch (Exception $e) {
                    echo '<div class="status-box error">';
                    echo "❌ <strong>Error:</strong> " . htmlspecialchars($e->getMessage());
                    echo '</div>';
                }
                echo '</div>';
                
                // Test 8: Color Prediction Generation Test
                echo '<div class="test-section">';
                echo '<h3>🎲 Color Prediction Generation Test</h3>';
                
                try {
                    // Test generation
                    $testResults = ColorPredictionGame::generateAllResults();
                    
                    echo '<div class="status-box success">';
                    echo "✅ <strong>Generation Test:</strong> Successful!<br>";
                    echo "<strong>Big:</strong> " . htmlspecialchars($testResults['big']) . " (should be 5-9)<br>";
                    echo "<strong>Small:</strong> " . htmlspecialchars($testResults['small']) . " (should be 0-4)<br>";
                    echo "<strong>Number:</strong> " . htmlspecialchars($testResults['number']) . " (should be 0-9)";
                    echo '</div>';
                    
                    // Test message formatting
                    $testPeriod = $db->getCurrentPeriodNumber();
                    $testMessage = ColorPredictionGame::formatMessage('big', $testResults['big'], $testPeriod);
                    
                    echo '<div class="status-box info">';
                    echo "<strong>Sample Message Format:</strong><br>";
                    echo "<div style='background: #f8f9fa; padding: 15px; border-radius: 5px; margin-top: 10px; font-family: monospace; white-space: pre-wrap;'>" . htmlspecialchars($testMessage) . "</div>";
                    echo '</div>';
                    
                } catch (Exception $e) {
                    echo '<div class="status-box error">';
                    echo "❌ <strong>Error:</strong> " . htmlspecialchars($e->getMessage());
                    echo '</div>';
                }
                echo '</div>';
                
                // Test 9: Database Methods Test
                echo '<div class="test-section">';
                echo '<h3>💾 Database Methods Test</h3>';
                
                $dbTests = [];
                
                // Test getCurrentPeriodNumber
                try {
                    $period = $db->getCurrentPeriodNumber();
                    $dbTests[] = ['name' => 'getCurrentPeriodNumber', 'status' => true, 'result' => "Period: #$period"];
                } catch (Exception $e) {
                    $dbTests[] = ['name' => 'getCurrentPeriodNumber', 'status' => false, 'result' => $e->getMessage()];
                }
                
                // Test predictionExistsForPeriod
                try {
                    $exists = $db->predictionExistsForPeriod($period);
                    $dbTests[] = ['name' => 'predictionExistsForPeriod', 'status' => true, 'result' => ($exists ? 'Exists' : 'Not exists')];
                } catch (Exception $e) {
                    $dbTests[] = ['name' => 'predictionExistsForPeriod', 'status' => false, 'result' => $e->getMessage()];
                }
                
                // Test getAllGroups
                try {
                    $groups = $db->getAllGroups();
                    $dbTests[] = ['name' => 'getAllGroups', 'status' => true, 'result' => count($groups) . ' groups'];
                } catch (Exception $e) {
                    $dbTests[] = ['name' => 'getAllGroups', 'status' => false, 'result' => $e->getMessage()];
                }
                
                // Test getAllAutoMessages
                try {
                    $messages = $db->getAllAutoMessages();
                    $dbTests[] = ['name' => 'getAllAutoMessages', 'status' => true, 'result' => count($messages) . ' messages'];
                } catch (Exception $e) {
                    $dbTests[] = ['name' => 'getAllAutoMessages', 'status' => false, 'result' => $e->getMessage()];
                }
                
                // Test getAutoMessagesToSend
                try {
                    $toSend = $db->getAutoMessagesToSend();
                    $dbTests[] = ['name' => 'getAutoMessagesToSend', 'status' => true, 'result' => count($toSend) . ' ready'];
                } catch (Exception $e) {
                    $dbTests[] = ['name' => 'getAutoMessagesToSend', 'status' => false, 'result' => $e->getMessage()];
                }
                
                // Display results
                echo '<table>';
                echo '<tr><th>Method</th><th>Status</th><th>Result</th></tr>';
                foreach ($dbTests as $test) {
                    $statusBadge = $test['status'] ? 'badge-success' : 'badge-danger';
                    $statusText = $test['status'] ? '✅ Pass' : '❌ Fail';
                    echo '<tr>';
                    echo '<td><code>' . htmlspecialchars($test['name']) . '</code></td>';
                    echo '<td><span class="badge ' . $statusBadge . '">' . $statusText . '</span></td>';
                    echo '<td>' . htmlspecialchars($test['result']) . '</td>';
                    echo '</tr>';
                }
                echo '</table>';
                echo '</div>';
                
                // Action Buttons
                echo '<div class="test-section">';
                echo '<h3>⚡ Quick Actions</h3>';
                echo '<div style="display: flex; gap: 10px; flex-wrap: wrap;">';
                echo '<a href="send_auto_messages.php" class="btn btn-success" target="_blank">▶️ Run Cron Job Now</a>';
                echo '<a href="?test_send=1" class="btn btn-success">📤 Test Send Message</a>';
                echo '<a href="../admin/auto_messages.php" class="btn">⚙️ Admin Panel</a>';
                echo '<a href="test_auto_messages.php" class="btn">🔄 Refresh</a>';
                echo '</div>';
                echo '</div>';
                
                // Test Send Message
                if (isset($_GET['test_send']) && $_GET['test_send'] == '1') {
                    echo '<div class="test-section">';
                    echo '<h3>📤 Test Message Send</h3>';
                    
                    if (!empty($allGroups)) {
                        $testGroup = $allGroups[0];
                        $testMsg = "🧪 <b>TEST MESSAGE</b>\n\nYe ek test message hai. Agar ye aapko dikh raha hai to system properly kaam kar raha hai! ✅";
                        
                        try {
                            $apiUrl = "https://api.telegram.org/bot" . BOT_TOKEN . "/sendMessage";
                            $data = [
                                'chat_id' => $testGroup['group_id'],
                                'text' => $testMsg,
                                'parse_mode' => 'HTML'
                            ];
                            
                            $ch = curl_init();
                            curl_setopt_array($ch, [
                                CURLOPT_URL => $apiUrl,
                                CURLOPT_RETURNTRANSFER => true,
                                CURLOPT_POST => true,
                                CURLOPT_POSTFIELDS => json_encode($data),
                                CURLOPT_HTTPHEADER => [
                                    'Content-Type: application/json'
                                ],
                                CURLOPT_SSL_VERIFYPEER => true,
                                CURLOPT_TIMEOUT => 10
                            ]);
                            
                            $response = curl_exec($ch);
                            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                            curl_close($ch);
                            
                            if ($httpCode === 200) {
                                $result = json_decode($response, true);
                                if (isset($result['ok']) && $result['ok'] === true) {
                                    echo '<div class="status-box success">';
                                    echo "✅ <strong>Test Message Sent Successfully!</strong><br>";
                                    echo "Group: <code>" . htmlspecialchars($testGroup['group_id']) . "</code><br>";
                                    echo "Message ID: " . htmlspecialchars($result['result']['message_id'] ?? 'N/A');
                                    echo '</div>';
                                } else {
                                    echo '<div class="status-box error">';
                                    echo "❌ <strong>Failed to send:</strong> " . htmlspecialchars($result['description'] ?? 'Unknown error');
                                    echo '</div>';
                                }
                            } else {
                                echo '<div class="status-box error">';
                                echo "❌ <strong>HTTP Error:</strong> $httpCode";
                                echo '</div>';
                            }
                        } catch (Exception $e) {
                            echo '<div class="status-box error">';
                            echo "❌ <strong>Error:</strong> " . htmlspecialchars($e->getMessage());
                            echo '</div>';
                        }
                    } else {
                        echo '<div class="status-box warning">';
                        echo "⚠️ <strong>No groups available</strong> for test. Bot ko pehle group mein add karo.";
                        echo '</div>';
                    }
                    echo '</div>';
                }
                
                echo '</div>';
                
                // Summary
                echo '<div class="test-section" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white;">';
                echo '<h3 style="color: white;">📋 Complete System Summary</h3>';
                echo '<div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 15px; margin-top: 15px;">';
                echo '<div style="background: rgba(255,255,255,0.1); padding: 15px; border-radius: 8px;">';
                echo '<strong>Database:</strong> ✅ Connected<br>';
                echo '<strong>Tables:</strong> ' . ($allTablesExist ? '✅ All exist' : '❌ Some missing');
                echo '</div>';
                echo '<div style="background: rgba(255,255,255,0.1); padding: 15px; border-radius: 8px;">';
                echo '<strong>Groups:</strong> ' . ($groupCount > 0 ? "✅ $groupCount registered" : '⚠️ None registered') . '<br>';
                echo '<strong>Auto Messages:</strong> ' . count($autoMessages) . ' total, ' . count($activeMessages) . ' active';
                echo '</div>';
                echo '<div style="background: rgba(255,255,255,0.1); padding: 15px; border-radius: 8px;">';
                echo '<strong>Color Prediction:</strong> ' . ($colorPredictionEnabled ? '✅ Enabled' : '⚠️ Disabled') . '<br>';
                echo '<strong>Cron Script:</strong> ' . ($cronExists ? '✅ Found' : '❌ Missing');
                echo '</div>';
                echo '</div>';
                echo '<div style="margin-top: 20px; padding: 15px; background: rgba(255,255,255,0.1); border-radius: 8px;">';
                echo '<strong>✅ All Tests Completed!</strong><br>';
                echo '<small>System is ready to use. Enable color prediction game from admin panel to start sending messages automatically.</small>';
                echo '</div>';
                echo '</div>';
                
            } catch (Exception $e) {
                echo '<div class="status-box error">';
                echo '❌ <strong>Error:</strong> ' . htmlspecialchars($e->getMessage());
                echo '</div>';
            }
            ?>
        </div>
    </div>
</body>
</html>
