<?php
/**
 * Webhook Setup Script
 * Run this once to set up the webhook with Telegram
 * 
 * Compatible with Telegram API format:
 * https://api.telegram.org/bot<TOKEN>/setWebhook?url=https://domain.com/webhook/bot.php
 */

require_once __DIR__ . '/config/config.php';

echo "Setting up Telegram webhook...\n\n";

// Check if bot token is configured
if (BOT_TOKEN === 'YOUR_BOT_TOKEN_HERE') {
    die("ERROR: Please configure BOT_TOKEN in config/config.php\n");
}

// Check if webhook URL is configured
if (WEBHOOK_URL === 'https://yourdomain.com/webhook/bot.php') {
    die("ERROR: Please configure WEBHOOK_URL in config/config.php with your HTTPS domain\n");
}

// Validate webhook URL format
if (!filter_var(WEBHOOK_URL, FILTER_VALIDATE_URL)) {
    die("ERROR: Invalid webhook URL format\n");
}

if (strpos(WEBHOOK_URL, 'https://') !== 0) {
    die("ERROR: Webhook URL must use HTTPS (Telegram requirement)\n");
}

$bot = new TelegramBot();

// Display configuration
echo "Configuration:\n";
echo "  Bot Token: " . substr(BOT_TOKEN, 0, 10) . "...\n";
echo "  Webhook URL: " . WEBHOOK_URL . "\n";
echo "  API Endpoint: https://api.telegram.org/bot" . substr(BOT_TOKEN, 0, 10) . ".../setWebhook\n";
echo "\n";

// Get current webhook info
echo "Current webhook info:\n";
$info = $bot->getWebhookInfo();
if ($info && isset($info['ok']) && $info['ok']) {
    $webhookInfo = $info['result'];
    echo "  URL: " . ($webhookInfo['url'] ?? 'Not set') . "\n";
    echo "  Pending updates: " . ($webhookInfo['pending_update_count'] ?? 0) . "\n";
    echo "\n";
}

// Set webhook using POST method (recommended)
echo "Setting webhook to: " . WEBHOOK_URL . "\n";
echo "Using API: https://api.telegram.org/bot<TOKEN>/setWebhook\n";
echo "\n";

$secretToken = (WEBHOOK_SECRET !== 'your-webhook-secret-token') ? WEBHOOK_SECRET : null;
$result = $bot->setWebhook(WEBHOOK_URL, $secretToken);

if ($result && isset($result['ok']) && $result['ok']) {
    echo "✅ Webhook set successfully!\n";
    echo "Description: " . ($result['description'] ?? 'OK') . "\n";
    echo "\n";
    echo "You can also set webhook manually using:\n";
    echo "https://api.telegram.org/bot" . BOT_TOKEN . "/setWebhook?url=" . urlencode(WEBHOOK_URL);
    if ($secretToken) {
        echo "&secret_token=" . urlencode($secretToken);
    }
    echo "\n";
} else {
    echo "❌ Failed to set webhook\n";
    if (isset($result['description'])) {
        echo "Error: " . $result['description'] . "\n";
    }
    if (isset($result['error_code'])) {
        echo "Error Code: " . $result['error_code'] . "\n";
    }
}

echo "\nDone!\n";
