<?php
/**
 * Telegram Webhook Handler
 * This file receives updates from Telegram
 */

require_once __DIR__ . '/../config/config.php';

// Verify webhook secret if configured
if (!empty(WEBHOOK_SECRET) && WEBHOOK_SECRET !== 'your-webhook-secret-token') {
    $secretHeader = $_SERVER['HTTP_X_TELEGRAM_BOT_API_SECRET_TOKEN'] ?? '';
    if ($secretHeader !== WEBHOOK_SECRET) {
        http_response_code(401);
        die('Unauthorized');
    }
}

// Read Telegram JSON updates
$update = json_decode(file_get_contents('php://input'), true);

if (!$update) {
    http_response_code(400);
    die('Invalid request');
}

// Log update (optional, for debugging)
// error_log("Update received: " . json_encode($update));

// Initialize database
$db = Database::getInstance();
$db->initDatabase();

// Initialize Telegram bot
$bot = new TelegramBot();

// Process update
try {
    $chatId = null;
    
    // Extract chat.id from different update types
    if (isset($update['message'])) {
        // Message update
        $chatId = $update['message']['chat']['id'] ?? null;
        $text = $update['message']['text'] ?? '';
        
        // Handle /start command
        if ($text === '/start' && $chatId) {
            if ($chatId < 0) {
                // Group/supergroup
                $chatTitle = $update['message']['chat']['title'] ?? 'Unknown';
                $response = "✅ Bot added to group: $chatTitle\n";
                $response .= "📋 Group ID: $chatId\n";
                $response .= "✅ Group registered successfully!";
                $bot->sendMessage($chatId, $response);
            } else {
                // Private message - show inline keyboard with buttons
                $response = "👋 <b>Welcome to Telegram Broadcast Bot!</b>\n\n";
                $response .= "📢 I can help you send messages to multiple groups and channels.\n\n";
                $response .= "👉 <b>Choose an option below:</b>";
                
                // Create inline keyboard with buttons
                $keyboard = [
                    'inline_keyboard' => [
                        [
                            [
                                'text' => '➕ Add My Group',
                                'callback_data' => 'add_group'
                            ],
                            [
                                'text' => '📢 Add My Channel',
                                'callback_data' => 'add_channel'
                            ]
                        ]
                    ]
                ];
                
                $bot->sendMessage($chatId, $response, 'HTML', json_encode($keyboard));
            }
        }
    } elseif (isset($update['my_chat_member'])) {
        // Bot added/removed from chat
        $chatId = $update['my_chat_member']['chat']['id'] ?? null;
        $newStatus = $update['my_chat_member']['new_chat_member']['status'] ?? '';
        
        if ($newStatus === 'member' && $chatId) {
            $chatTitle = $update['my_chat_member']['chat']['title'] ?? 'Unknown';
            $response = "✅ Bot added to group: $chatTitle\n";
            $response .= "📋 Group ID: $chatId\n";
            $response .= "✅ Group registered successfully!";
            $bot->sendMessage($chatId, $response);
        }
    } elseif (isset($update['edited_message'])) {
        // Edited message
        $chatId = $update['edited_message']['chat']['id'] ?? null;
    } elseif (isset($update['channel_post'])) {
        // Channel post
        $chatId = $update['channel_post']['chat']['id'] ?? null;
    } elseif (isset($update['callback_query'])) {
        // Handle callback query (inline keyboard button clicks)
        $callbackQuery = $update['callback_query'];
        $callbackData = $callbackQuery['data'] ?? '';
        $callbackChatId = $callbackQuery['message']['chat']['id'] ?? null;
        $callbackQueryId = $callbackQuery['id'] ?? '';
        $userId = $callbackQuery['from']['id'] ?? null;
        
        if ($callbackData === 'add_group') {
            // Answer callback query first
            $bot->answerCallbackQuery($callbackQueryId, 'Instructions for adding group');
            
            // Get bot username
            $botInfo = $bot->getMe();
            $botUsername = '';
            if ($botInfo && isset($botInfo['ok']) && $botInfo['ok'] && isset($botInfo['result']['username'])) {
                $botUsername = '@' . $botInfo['result']['username'];
            } else {
                $botUsername = 'this bot';
            }
            
            $instructions = "📋 <b>How to Add Your Group:</b>\n\n";
            $instructions .= "1️⃣ Go to your Telegram group\n";
            $instructions .= "2️⃣ Click on group name → <b>Add Members</b>\n";
            $instructions .= "3️⃣ Search for: <code>" . $botUsername . "</code>\n";
            $instructions .= "4️⃣ Select the bot and add it\n";
            $instructions .= "5️⃣ Send <code>/start</code> in the group\n\n";
            $instructions .= "✅ The group will be automatically registered!";
            
            $bot->sendMessage($callbackChatId, $instructions, 'HTML');
        } elseif ($callbackData === 'add_channel') {
            // Answer callback query first
            $bot->answerCallbackQuery($callbackQueryId, 'Instructions for adding channel');
            
            // Get bot username
            $botInfo = $bot->getMe();
            $botUsername = '';
            if ($botInfo && isset($botInfo['ok']) && $botInfo['ok'] && isset($botInfo['result']['username'])) {
                $botUsername = '@' . $botInfo['result']['username'];
            } else {
                $botUsername = 'this bot';
            }
            
            $instructions = "📢 <b>How to Add Your Channel:</b>\n\n";
            $instructions .= "1️⃣ Go to your Telegram channel\n";
            $instructions .= "2️⃣ Click on channel name → <b>Administrators</b>\n";
            $instructions .= "3️⃣ Click <b>Add Administrator</b>\n";
            $instructions .= "4️⃣ Search for: <code>" . $botUsername . "</code>\n";
            $instructions .= "5️⃣ Select the bot and add as admin\n";
            $instructions .= "6️⃣ The channel will be automatically registered!\n\n";
            $instructions .= "⚠️ <b>Note:</b> Bot must be admin to send messages to channels.";
            
            $bot->sendMessage($callbackChatId, $instructions, 'HTML');
        }
    }
    
    // If chat_id < 0 → treat as group/supergroup and store
    if ($chatId !== null && $chatId < 0) {
        // Store group ID using INSERT IGNORE (via PDO prepared statement)
        $db->saveGroup((string)$chatId);
    }
    
    // Respond with 200 OK
    http_response_code(200);
    echo 'OK';
    
} catch (Exception $e) {
    error_log("Webhook error: " . $e->getMessage());
    http_response_code(500);
    echo 'Error';
}
